#ifndef _MESSAGEPIPE_H
#define _MESSAGEPIPE_H

namespace UnityPlugin
{
	namespace Messages
	{
		enum MessageType
		{
			kNPToolKit_NotSet,
			kNPToolKit_Log,
			kNPToolKit_LogWarning,
			kNPToolKit_LogError,

			kNPToolKit_NPInitialized,		// NP system initialization has completed.

			kNPToolKit_ConnectionUp,		// Connected to network.
			kNPToolKit_ConnectionDown,		// Network connection was lost.
			kNPToolKit_SignedIn,			// Signed into Playstation Network.
			kNPToolKit_SignedOut,			// Signed out of Playstation Network.
			kNPToolKit_SignInError,			// An error occurred.

			kNPToolKit_SysEvent,			// General system events. Event id is passed in message data.
			kNPToolKit_SysResume,			// Application resumed after being suspended.
			kNPToolKit_SysNpMessageArrived,	// An NP invite or data message was received.
			kNPToolKit_SysStorePurchase,	// A product was purchased at the PlayStationStore.
			kNPToolKit_SysStoreRedemption,	// A promotion code was redeemed at the PlaystationStore.
			
			kNPToolKit_GotUserProfile,			// Retrieval of user profile data has completed.
			kNPToolKit_GotRemoteUserNpID,		// Retrieval of other users NpID has completed.
			kNPToolKit_GotRemoteUserProfile,	// Retrieval of other users profile data has completed.
			kNPToolKit_UserProfileError,		// An error occurred while retrieving profile data.

			kNPToolKit_NetInfoGotBandwidth,		// Retrieval of network bandwidth data has completed.
			kNPToolKit_NetInfoGotBasic,			// Retrieval of basic network info has completed.
			kNPToolKit_NetInfoError,			// An error occurred.

			kNPToolKit_FriendsListUpdated,  // Friends list has been updated, you should request the list to get the new version.
			kNPToolkit_FriendsPresenceUpdated,	// Friends presence has changed - going online/offline
			kNPToolKit_GotFriendsList,		// Retrieval of friends list has completed.
			kNPToolKit_FriendsListError,	// An error occurred while retrieving friends data.
			kNPToolKit_PresenceSet,			// Online presence has been set.
			kNPToolKit_PresenceError,		// An error occurred while getting or setting online presence.

			kNPToolKit_TrophySetSetupSuccess,		// Trophy pack installation succeeded.
			kNPToolKit_TrophySetSetupCancelled,		// Trophy pack installation was canceled.
			kNPToolKit_TrophySetSetupAborted,		// Trophy pack installation was aborted.
			kNPToolKit_TrophySetSetupFail,			// Trophy pack installation failed.
			kNPToolKit_TrophyCacheReady,			// Trophy cache is ready.
			kNPToolKit_TrophyGotGameInfo,			// Trophy game info has been cached.
			kNPToolKit_TrophyGotGroupInfo,			// Trophy group info has been cached.
			kNPToolKit_TrophyGotTrophyInfo,			// Trophy info has been cached.
			kNPToolKit_TrophyGotProgress,			// Trophy progress has been cached.
			kNPToolKit_TrophyUnlocked,				// Trophy was unlocked.
			kNPToolKit_TrophyUnlockFailed,			// Trophy unlock failed.
			kNPToolKit_TrophyUnlockedAlready,		// Trophy has already been unlocked.
			kNPToolKit_TrophyUnlockedPlatinum,		// Platinum trophy has been unlocked.
			kNPToolKit_TrophyError,					// An error occurred.

			kNPToolKit_RankingCacheRegistered,		// The ranking cache has been registered.
			kNPToolKit_RankingNewBestScore,			// A new best score has been registered.
			kNPToolKit_RankingNotBestScore,			// Registered score was not the best score.
			kNPToolKit_RankingGotOwnRank,			// Own rank has been retrieved.
			kNPToolKit_RankingGotFriendRank,		// Friend rank has been retrieved.
			kNPToolKit_RankingGotRankList,			// Rank list has been retrieved.
			kNPToolKit_RankingError,				// An error occurred.

			kNPToolKit_MatchingCreatedSession,		// A session has been created.
			kNPToolKit_MatchingFoundSessions,		// Found sessions.
			kNPToolKit_MatchingJoinedSession,		// A session has been joined.
			kNPToolKit_MatchingJoinInvalidSession,	// Attempted to join a session but the session does not exists.
			kNPToolKit_MatchingUpdatedSession,		// The current session information has changed.
			kNPToolKit_MatchingLeftSession,			// Left a session.
			kNPToolKit_MatchingRoomDestroyed,		// Session room was destroyed, you need to disconnect.
			kNPToolKit_MatchingKickedOut,			// Kicked out of session, you need to disconnect.
			kNPToolKit_MatchingError,				// An error occurred.

			kNPToolKit_WordFilterNotCensored,		// Word filter allowed the comment.
			kNPToolKit_WordFilterCensored,			// Word filter disallowed the comment.
			kNPToolKit_WordFilterSanitized,			// Word filter has sanitized the comment (it may not have changed if it was already good).
			kNPToolKit_WordFilterError,				// An error occurred.

			kNPToolKit_MessagingSent,				// The message was sent.
			kNPToolKit_MessagingCanceled,			// The message send was canceled by the user.
			kNPToolKit_MessagingNotSent,			// The message was not sent, reason unknown.
			kNPToolKit_MessagingNotSentFreqTooHigh,			// The message was not sent because the sending frequency is too high.
			kNPToolKit_MessagingSessionInviteRetrieved,		// A session invite message has been retrieved.
			kNPToolKit_MessagingCustomInviteRetrieved,	// A custom invite message has been retrieved.
			kNPToolKit_MessagingDataMessageRetrieved,	// A custom data message has been retrieved.
			kNPToolKit_MessagingInGameDataMessageRetrieved,	// An in-game data message has been retrieved.
			kNPToolKit_MessagingSessionInviteReceived, // A session invite has been received by the user.
			kNPToolKit_MessagingSessionInviteAccepted, // A session invite has been accepted by the user.

			kNPToolKit_CommerceSessionCreated,			// A commerce session has been created.
			kNPToolKit_CommerceSessionAborted,			// Commerce session creation has been aborted.
			kNPToolKit_CommerceGotCategoryInfo,			// Category info has been retrieved.
			kNPToolKit_CommerceGotProductList,			// Product list has been retrieved.
			kNPToolKit_CommerceGotProductInfo,			// Product info has been retrieved
			kNPToolKit_CommerceGotEntitlementList,		// Entitlements list has been retrieve.
			kNPToolKit_CommerceConsumedEntitlement,		// An entitlement has been consumed.
			kNPToolKit_CommerceError,					// An event generated when a commerce error has occurred.
			kNPToolKit_CommerceCheckoutStarted,			// Checkout dialog has started.
			kNPToolKit_CommerceCheckoutFinished,		// Checkout dialog has finished.

			kNPToolKit_TUSDataSet,					// TUS Data has been set.
			kNPToolKit_TUSDataReceived,				// TUS Data has been received.
			kNPToolKit_TUSVariablesSet,				// TUS Variables have been set.
			kNPToolKit_TUSVariablesModified,		// TUS Variables have been modified.
			kNPToolKit_TUSVariablesReceived,		// TUS Variables have been received.
			kNPToolKit_TSSDataReceived,				// TSS Data has been received.
			kNPToolKit_TSSNoData,					// There is no TSS data.
			kNPToolKit_TusTssError,					// An error occurred.

			kNPToolKit_DlgFriendsClosed,				// The friends dialog has been closed.
			kNPToolKit_DlgSharedPlayHistoryClosed,		// The shared play history dialog has been closed.
			kNPToolKit_DlgProfileClosed,				// The profile dialog has been closed.
			kNPToolKit_DlgCommerceClosed,				// The commerce dialog has been closed.

			kNPToolKit_FacebookDialogStarted,			// Facebook dialog started.
			kNPToolKit_FacebookDialogFinished,			// Facebook dialog finished.
			kNPToolKit_FacebookMessagePosted,			// Successfully posted a message to facebook.
			kNPToolKit_FacebookMessagePostFailed,		// Failed to  posted a message to facebook.

			kNPToolKit_TwitterDialogStarted,			// Twitter dialog started.
			kNPToolKit_TwitterDialogFinished,			// Twitter dialog finished.
			kNPToolKit_TwitterDialogCanceled,			// Twitter dialog canceled.
			kNPToolKit_TwitterMessagePosted,			// Successfully posted a message to twitter.
			kNPToolKit_TwitterMessagePostFailed,		// Failed to  posted a message to twitter.


			kNPToolKit_TicketingGotTicket,				// Got an auth ticket.
			kNPToolKit_TicketingError,					// An error occurred.

			kNPToolKit_CheckPlusResult,					// CheckPlus request result
			kNPToolKit_AccountLanguageResult,			// GetAccountLanguage request result
			kNPToolKit_ParentalControlResult,			// GetParentalControlInfo request result
		};

		enum 
		{
			kBufferSize = 1024
		};

		struct PluginMessage
		{
			PluginMessage()
			{
				type = kNPToolKit_NotSet;
				dataSize = 0;
				data = NULL;
			}
			~PluginMessage()
			{
				if(dataSize > 0)
				{
					free(data);
				}
			}

			void SetData(void* src, int size)
			{
				dataSize = size;
				data = (char*)malloc(dataSize);
				memcpy(data, src, dataSize);
			}

			template<typename T> void SetData(const T& val)
			{
				Assert(sizeof(val) == sizeof(void*));
				dataSize = 0;
				memcpy(&data, &val, sizeof(void*));
			}

			void SetDataFromString(char* src)
			{
				int len = (int)strlen(src);
				dataSize = len + 1;
				data = malloc(dataSize);
				memcpy(data, src, dataSize);
			}

			// Message type.
			int type;

			// Message data.
			// NOTE, if dataSize != 0 then the data was allocated, if dataSize == 0 then the data void* is actually some 4byte blittable value, i.e, an int.
			int dataSize;
			void* data;
		};

		void Log(const char* format, ...);
		void LogWarning(const char* format, ...);
		void LogError(const char* format, ...);
		void AddMessage(MessageType msgType);
		void AddMessage(PluginMessage* msg);
		bool HasMessage();
		bool GetFirstMessage(PluginMessage* msg);
		bool RemoveFirstMessage();
	};

	PRX_EXPORT bool PrxHasMessage();
	PRX_EXPORT bool PrxGetFirstMessage(Messages::PluginMessage* result);
	PRX_EXPORT bool PrxRemoveFirstMessage();
}

#if NDEBUG
#define DBG_LOG(...)
#define DBG_LOG_WARNING(...)
#define DBG_LOG_ERROR(...)
#else
#define DBG_LOG(...) UnityPlugin::Messages::Log(__VA_ARGS__);
#define DBG_LOG_WARNING(...) UnityPlugin::Messages::LogWarning(__VA_ARGS__);
#define DBG_LOG_ERROR(...) UnityPlugin::Messages::LogError(__VA_ARGS__);
#endif

#endif // _MESSAGEPIPE_H
